<?php
/*
	* E-Mail Notifier: a small PHP-GTK2 application which queries POP3/IMAP servers
	*if the user has new unread mail
	* Copyright (C) 2008  Szigeti "BlackY" Marton

	* This program is free software: you can redistribute it and/or modify
	* it under the terms of the GNU General Public License as published by
	* the Free Software Foundation, either version 3 of the License, or
	* (at your option) any later version.

	* This program is distributed in the hope that it will be useful,
	* but WITHOUT ANY WARRANTY; without even the implied warranty of
	* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	* GNU General Public License for more details.

	* You should have received a copy of the GNU General Public License
	* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
	* E-mail notifier
	*
	* This file contains all the classes/definitions E-mail notifier needs
	* to run.
	* @author Szigeti "BlackY" Márton <blacky@invitel.hu>
	* @version 1.0
	* @package emailn
*/
/**#@+
#
 * Constants
#
 */
/**
	* This constant contains the message shown to the user when a server
	* is not reachable or an authentication error occured.
*/
define('ERR_CONNECT', 'Failed to connect to %1$s! Try again?');
/**
	* This constant contains the message shown to the user when a client
	* could not be started
*/
define('ERR_CLIENT', 'Failed to start client, because: %1$s'); 
/**
	* This constant contains the title of Email Notifiers window
*/
define('LANG_TITLE', 'Mail Notifier');
/**
	* This constant contains the caption of an "OK" button
*/
define('LANG_OK', 'OK');
/**
	* This constant contains the caption of a "Cancel" button
*/
define('LANG_CANCEL', 'Cancel');

/**
	* This is the main class of Mail Notifier, handling almost anything.
	* @package emailn
	* @subpackage classes
*/
Class MailNotifierWindow extends GTKWindow {
	/**
		* This is an array of MailNotifierButton instances (1 per mailbox)
		* @access protected
		* @var MailNotifierButton
	*/
	protected $buttons = array();
	/**
		* This is the GTKHButtonBox instance used to add the buttons to the window
		* @access protected
		* @var GTKHButtonBox
	*/
	protected $buttonbox;
	
	/**
		* This constant should be used if you want a Horizontal button bar
	*/
	const HORIZONTAL = 0;
	/**
		* This constant should be used if you want a Vertical button bar
	*/
	const VERTICAL = 1;
	
	/**
		* Object constructor
		* It sets up the GUI elements based on the array of mailboxes given
		* and starts the GTK timers. (1 per mailbox)
		* @param Array $mailbox
		* @return void
		* @access public
	*/
	public function __Construct(Array $mailboxes, $orientation = MailNotifierWindow::HORIZONTAL) {
		/* Call parents' constructor */
		parent::__Construct(Gtk::WINDOW_TOPLEVEL);
		/* Set up window style:
			- Resize to 1*1 pixel (smallest size possible)
			- Remove title bar
			- Set the title (shown in taskbar)
			- Try to set an icon (mail.png)
		*/
		$this->resize(1,1);
		$this->set_decorated(false);
		$this->set_title(LANG_TITLE);
		try { $this->set_icon_from_file('mail.png'); } catch(Exception $e) { }
		/* Create a button box based on $orientation */
		$class = ($orientation == MailNotifierWindow::VERTICAL ? 'GTKVButtonBox' : 'GTKHButtonBox');
		$this->buttonbox = new $class;
		unset($class);
		$this->add($this->buttonbox);
		/* Create one button for each mailbox */
		foreach($mailboxes as $mailboxId => $mailboxData) {
			/* Check mailbox data validity */
			if(!IsSet($mailboxData['Interval'])
				|| ((!IsSet($mailboxData['Host']) || !IsSet($mailboxData['Port'])) && !(IsSet($mailboxData['IMAPStr'])))
				|| !IsSet($mailboxData['Client'])) continue;
			/* Create the button, (load an icon if possible) and create a timer for the mailbox */
			$this->buttons[$mailboxId] = new MailNotifierButton($mailboxData);
			$this->buttons[$mailboxId]->connect_simple('released', array(&$this, 'button_onreleased'), $mailboxId);
			if(IsSet($mailboxData['Icon'])) {
				try {
					$image = GTKImage::new_from_file(dirname(__FILE__) . '/' . $mailboxData['Icon']);
					$this->buttons[$mailboxId]->set_image($image);
				}
				catch(Exception $e) { }
			}
			if($this->checkmail($mailboxId)) {
				Gtk::timeout_add($mailboxData['Interval'] * 60000, array(&$this, 'timer_ontimeout'), $mailboxId);
			}
		}
		return true;
	}
	
	/**
		* This method shows the button with id $mailboxId
		* @param string $mailboxId The ID of the mailbox (section name in .ini)
		* @return void
		* @access protected
	*/
	protected function button_show( $mailboxId ) {
		if($this->buttons[$mailboxId]->shown) return;
		$this->buttonbox->add($this->buttons[$mailboxId]);
		$this->buttons[$mailboxId]->shown = true;
		$this->show_all();
	}

	/**
		* This method hides the button with id $mailboxId
		* @param string $mailboxId The ID of the mailbox (section name in .ini)
		* @return void
		* @access protected
	*/	
	protected function button_hide( $mailboxId ) {
		$this->buttonbox->remove($this->buttons[$mailboxId]);
		$this->buttons[$mailboxId]->shown = false;
		if(!$this->buttonbox->get_children()) {
			$this->hide_all();
		}
	}
	
	/**
		* This method is called when any of the buttons are clicked
		* It removes the buttons with the same Client as the one clicked (incl. the clicked one)
		*and starts the associated client with a COM object (WScript.Shell -> run)
		* @param string $mailboxId The ID of the mailbox (section name in .ini)
		* @return void
		* @access public
	*/
	public function button_onreleased( $mailboxId ) {
		/* Find buttons with the same client associated... */
		foreach($this->buttons as $id => $button) {
			if(!$button->shown) continue;
			if($button->mailboxData['Client'] == $this->buttons[$mailboxId]->mailboxData['Client']) {
				/* ... and hide them */
				$this->button_hide($id);
			}
		}
		/* Resize window, so no blank space will be shown */
		$this->resize(1,1);
		/* Start the program */
		$shell = new COM('WScript.Shell');
		try {
			$errorlevel = $shell->Run('"' . $this->buttons[$mailboxId]->mailboxData['Client'] . '"');
		}
		catch(Exception $e) {
			/* An error occured: show a dialog telling this to the user */
			$msg = new GTKMessageDialog($this, 0, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, sprintf(ERR_CLIENT, $e->getMessage()));
			$msg->run();
			$msg->destroy();
		}
	}

	/**
		* This function is called when a mailboxs timer times out.
		* @param string $mailboxId The ID of the mailbox (section name in .ini)
		* @param string 
	*/
	public function timer_ontimeout( $mailboxId ) {
		return $this->checkmail($mailboxId);
	}
	
	/**
		* This function logs into a POP3/IMAP server and checks wheter there're unread mail
		* It returns false if an error occured and the user chose not to continue fetching mail info
		* from this server, true otherwise.
		* @param string $mailboxId The ID of the mailbox (section name in .ini)
		* @return bool
	*/
	public function checkmail( $mailboxId ) {
		$mailboxData = $this->buttons[$mailboxId]->mailboxData;
		$imapstr = IsSet($mailboxData['IMAPStr']) ? $mailboxData['IMAPStr'] : "{" . $mailboxData['Host'] . ":" . $mailboxData['Port'] . "/pop3}";
		if(($imap = @imap_open($imapstr, $mailboxData['Username'], $mailboxData['Password'], 0, 1)) == false) {
			$msgBox = new GtkMessageDialog($this, Gtk::DIALOG_MODAL, Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO, sprintf(ERR_CONNECT, $mailboxId));
			$response = $msgBox->run();
			$msgBox->destroy();
			switch($response) {
				case Gtk::RESPONSE_YES:
					return true;
					break;
				case Gtk::RESPONSE_NO:
					return false;
					break;
			}
		}
		$status = imap_status($imap, $imapstr . "INBOX", SA_UNSEEN);
		if($status->unseen) {
			$this->buttons[$mailboxId]->set_label($mailboxId . ' (' . $status->unseen . ')');
			$this->button_show($mailboxId);
		}
		imap_close($imap);
		return true;
	}
	
	
}

/**
	* This class is a GTKButton with two new properties
	* @package emailn
	* @subpackage classes
*/	
Class MailNotifierButton extends GTKButton {
	/**
		* This variable stores the mail box data (host, port etc.)
		* @var array
	*/
	public $mailboxData = array();
	
	/**
		* This variable is set if the button is currently visible
		* (eg.: added to a ButtonBox instance)
		* @var bool
	*/
	public $shown = false;
	
	/**
		* Object constructor.
		* Creates a MailNotifierButton instance and saves $mailboxData
		* @param array $mailboxData Host, port etc. fetched from .ini
		* @return void
	*/
	public function __Construct(Array $mailboxData) {
		parent::__Construct();
		$this->mailboxData = $mailboxData;
	}

}


/* Command line parameters, etc. */
$gplText = <<<LICENSE
    Mail Notifier  Copyright (C) 2008  Szigeti Marton
LICENSE;
print $gplText;
/* Parse command line parameters */
$argv = array_flip($argv);
if(IsSet($argv['/?'])) {
	$infoText = <<<INFO
	Command line parameters:
	/?	- This help info
	-v	- Set orientation to vertical
INFO;
	die($infoText);
}
/* Orientation has been added? */
$orientation = (IsSet($argv['-v']) || IsSet($argv['-V'])) ? MailNotifierWindow::VERTICAL : MailNotifierWindow::HORIZONTAL;
/* Create the MailNotifierWindow instance */
$mcw = new MailNotifierWindow(parse_ini_file('mailboxes.ini', 1), $orientation);
/* And start GTKs main loop */
Gtk::main();